"""Utility functions for the currency conversion service.

Provides helpers for validating currency codes and fetching exchange rates.
Rates are obtained from Fawaz Ahmed's open-source API with a fallback to
exchangerate.host, and a static final fallback.
"""

from __future__ import annotations

from typing import Dict
import requests


FAWAZ_BASE = "https://cdn.jsdelivr.net/gh/fawazahmed0/currency-api@latest/v1"
ERH_BASE = "https://api.exchangerate.host"

# Minimal static fallback so the API never completely dies if offline.
STATIC_FALLBACK_RATES: Dict[str, Dict[str, float]] = {
    "usd": {"usd": 1.0, "ngn": 1500.0, "eur": 0.9},
    "ngn": {"ngn": 1.0, "usd": 1 / 1500.0, "eur": 0.9 / 1500.0},
}


def validate_currency(code: str) -> str:
    """Basic validation & normalization for ISO currency codes.

    Raises:
        ValueError: If the code is missing or malformed.
    """
    if not code or not isinstance(code, str):
        raise ValueError("Currency code is required")
    c = code.strip().lower()
    if len(c) != 3 or not c.isalpha():
        raise ValueError(f"Invalid currency code '{code}'")
    return c


def fetch_rates(base: str) -> Dict[str, float]:
    """Fetch conversion rates for a base currency.

    The result maps target currency (lowercase) to numeric rate.
    """
    base = base.lower()

    # Try Fawaz Ahmed's API
    try:
        url = f"{FAWAZ_BASE}/currencies/{base}.json"
        r = requests.get(url, timeout=5)
        if r.ok:
            data = r.json()
            if base in data:
                raw = data[base]
                return {k.lower(): float(v) for k, v in raw.items()}
    except Exception:
        pass

    # Fallback: exchangerate.host
    try:
        url = f"{ERH_BASE}/latest"
        r = requests.get(url, params={"base": base.upper()}, timeout=5)
        if r.ok:
            data = r.json()
            raw_rates = data.get("rates")
            if raw_rates:
                return {k.lower(): float(v) for k, v in raw_rates.items()}
    except Exception:
        pass

    # Final fallback: static mapping or self‑rate
    if base in STATIC_FALLBACK_RATES:
        return STATIC_FALLBACK_RATES[base]
    return {base: 1.0}
